const clickedElementType = {
    Self:10,
    Child:20,
    Other:99
}

let listenersAdded = false;

let closableComponents = [];

function getClickedElementType(clickedElement, closableElement) {
    if (clickedElement === closableElement)
        return clickedElementType.Self;

    let currentElement = clickedElement;

    do {
        if (currentElement.parentElement === closableElement)
            return clickedElementType.Child;
        currentElement = currentElement.parentElement;
    } while (currentElement !== null);

    return clickedElementType.Other;
}

function addClosableComponent(element, dotnetAdapter) {
    closableComponents.push({element: element, dotnetAdapter: dotnetAdapter});
}

function findClosableComponent(element) {
    for (let index = 0; index < closableComponents.length; ++index) {
        if (closableComponents[index].element === element)
            return closableComponents[index];
    }
    return null;
}

function findClosableComponentIndex(element) {
    for (let index = 0; index < closableComponents.length; ++index) {
        if (closableComponents[index].element === element)
            return index;
    }
    return -1;
}

function isClosableComponent(element) {
    for (let index = 0; index < closableComponents.length; ++index) {
        if (closableComponents[index].element === element)
            return true;
    }
    return false;
}

function registerClosableComponent(element, dotnetAdapter) {
    if (element) {
        if (isClosableComponent(element) !== true) {
            addClosableComponent(element, dotnetAdapter);
        }
    }
}

function unregisterClosableComponent(element) {
    if (element) {
        const index = findClosableComponentIndex(element);
        if (index !== -1) {
            closableComponents.splice(index, 1);
        }
    }
}

function closeComponentIfRequired(clickedElement) {
    if (closableComponents && closableComponents.length > 0) {
        const lastClosable = closableComponents[closableComponents.length - 1];

        if (clickedElement) {
            const elementType = getClickedElementType(clickedElement, lastClosable.element);
            
            if (elementType === clickedElementType.Self
                || elementType === clickedElementType.Child) {
                return;
            }
        }

        if (lastClosable) {
            lastClosable.dotnetAdapter.invokeMethodAsync('Close');
        }
    }
}

function onDocumentClick(event) {
    closeComponentIfRequired(event.target);
}

function onDocumentKeyUp(event) {
    if (event.keyCode === 27) {
        closeComponentIfRequired(null);
    }
}

function addListeners() {
    document.addEventListener(
        'click',
        onDocumentClick,
        // We have to call this listener before any other event:
        /* useCapture: */ true);

    document.addEventListener(
        'keyup',
        onDocumentKeyUp);
}

function removeListeners() {
    document.removeEventListener(
        'click',
        onDocumentClick,
        /* useCapture: */ true);

    document.removeEventListener(
        'keyup',
        onDocumentKeyUp);
}

function getSingleElementByTagName(elementTagName) {
    let elements = document.getElementsByTagName(elementTagName);
    
    if (elements.length !== 1) {
        throw new Error('Couldn\'t find single element by tag name "' + elementTagName + '".');
    }

    return elements[0];
}

function containsCssClassByTagName(elementTagName, cssClass) {
    let element = getSingleElementByTagName(elementTagName);

    return element.classList.contains(cssClass);
}

function addCssClassByTagName(elementTagName, cssClass) {
    let element = getSingleElementByTagName(elementTagName);

    if (!element.classList.contains(cssClass)) {
        element.classList.add(cssClass);
    }
}

function removeCssClassByTagName(elementTagName, cssClass) {
    let element = getSingleElementByTagName(elementTagName);

    if (element.classList.contains(cssClass)) {
        element.classList.remove(cssClass);
    }
}

export {
    registerClosableComponent,
    unregisterClosableComponent,
    addListeners,
    removeListeners,
    containsCssClassByTagName,
    addCssClassByTagName,
    removeCssClassByTagName
}